import 'dart:convert';
import 'package:flutter/material.dart';
import 'package:flutter_riverpod/flutter_riverpod.dart';
import 'package:fluttertoast/fluttertoast.dart';
import 'package:zcart_delivery/models/order_light_model.dart';
import 'package:zcart_delivery/network/api.dart';
import 'package:zcart_delivery/network/network_helper.dart';
import 'package:zcart_delivery/translations/locale_keys.g.dart';
import 'package:easy_localization/easy_localization.dart';
import 'package:zcart_delivery/models/order_model.dart';

final ordersProvider = FutureProvider<List<OrderLightModel>?>((ref) async {
  final response = await NetworkHelper.getRequest(
    API.orders,
    bearerToken: true,
    requestBody: {"filter": "assigned"},
  );

  if (response != null) {
    debugPrint(response.body);
    if (isRequestSuccessful(response.statusCode)) {
      List<OrderLightModel> orders = [];

      for (var order in jsonDecode(response.body)['data']) {
        orders.add(OrderLightModel.fromJson(order));
      }

      return orders;
    } else if (response.statusCode == 401) {
      Fluttertoast.showToast(msg: LocaleKeys.you_are_not_logged_in.tr());

      return null;
    } else {
      Fluttertoast.showToast(
          msg: jsonDecode(response.body)['message'] ??
              LocaleKeys.something_went_wrong.tr());
      return [];
    }
  } else {
    Fluttertoast.showToast(msg: LocaleKeys.something_went_wrong.tr());
    return [];
  }
});

final orderProvider =
    FutureProvider.autoDispose.family<OrderModel?, int>((ref, orderId) async {
  try {
    final response =
        await NetworkHelper.getRequest(API.order(orderId), bearerToken: true);

    if (response != null) {
      debugPrint(response.body);
      if (isRequestSuccessful(response.statusCode)) {
        final order = OrderModel.fromJson(jsonDecode(response.body)['data']);

        return order;
      } else if (response.statusCode == 401) {
        Fluttertoast.showToast(msg: LocaleKeys.you_are_not_logged_in.tr());

        return null;
      } else {
        Fluttertoast.showToast(
            msg: jsonDecode(response.body)['message'] ??
                LocaleKeys.something_went_wrong.tr());
        return null;
      }
    } else {
      Fluttertoast.showToast(msg: LocaleKeys.something_went_wrong.tr());
      return null;
    }
  } catch (e) {
    Fluttertoast.showToast(msg: e.toString());
    return null;
  }
});

class OrderController {
  static Future<bool> updateStatus(int orderId, int status) async {
    try {
      final response = await NetworkHelper.postRequest(
          API.updateStatus(orderId.toString()), {},
          bearerToken: true);

      if (response != null) {
        debugPrint(response.body);
        if (isRequestSuccessful(response.statusCode)) {
          if (jsonDecode(response.body)['isSuccess'] == true) {
            Fluttertoast.showToast(msg: jsonDecode(response.body)['message']);
          }
          return jsonDecode(response.body)['isSuccess'] ?? false;
        } else {
          Fluttertoast.showToast(
              msg: jsonDecode(response.body)['message'] ??
                  LocaleKeys.something_went_wrong.tr());

          return false;
        }
      } else {
        Fluttertoast.showToast(msg: LocaleKeys.something_went_wrong.tr());
        return false;
      }
    } catch (e) {
      Fluttertoast.showToast(msg: e.toString());
      return false;
    }
  }

  static Future<bool> updatePaymentStatus(int orderId) async {
    // Map<String, String> _requestBody = {'payment_status': 1.toString()};

    try {
      final response = await NetworkHelper.postRequest(
          API.updatePaymentStatus(orderId.toString()), {},
          bearerToken: true);

      if (response != null) {
        debugPrint(response.body);
        if (isRequestSuccessful(response.statusCode)) {
          if (jsonDecode(response.body)['isSuccess'] == true) {
            Fluttertoast.showToast(msg: jsonDecode(response.body)['message']);
          }
          return jsonDecode(response.body)['isSuccess'] ?? false;
        } else {
          Fluttertoast.showToast(
              msg: jsonDecode(response.body)['message'] ??
                  LocaleKeys.something_went_wrong.tr());

          return false;
        }
      } else {
        Fluttertoast.showToast(msg: LocaleKeys.something_went_wrong.tr());
        return false;
      }
    } catch (e) {
      Fluttertoast.showToast(msg: e.toString());
      return false;
    }
  }

  // Send OTP
  static Future<bool> sendOtp(int orderId, int otp) async {
    try {
      final response = await NetworkHelper.postRequest(
        API.sendOtp(orderId.toString()),
        {'otp': otp.toString()},
        bearerToken: true,
      );

      if (response != null) {
        debugPrint(response.body);
        if (isRequestSuccessful(response.statusCode)) {
          if (jsonDecode(response.body)['isSuccess'] == true) {
            Fluttertoast.showToast(msg: jsonDecode(response.body)['message']);
          }
          return true;
        } else {
          Fluttertoast.showToast(
              msg: jsonDecode(response.body)['message'] ??
                  LocaleKeys.something_went_wrong.tr());
          return false;
        }
      } else {
        Fluttertoast.showToast(msg: LocaleKeys.something_went_wrong.tr());
        return false;
      }
    } catch (e) {
      Fluttertoast.showToast(msg: e.toString());
      return false;
    }
  }
}
